unit MainForm;

interface

uses
  tb.AppLogger,
  tb.Open62541.Types,
  tb.Open62541.StatusCodes,
  tb.Open62541.NodeIds,
  tb.Open62541.Functions,

  Globals,

  Winapi.Windows,
  Winapi.Messages,

  System.SysUtils,
  System.Variants,
  System.Classes,

  Vcl.Graphics,
  Vcl.Controls,
  Vcl.Forms,
  Vcl.Dialogs,
  Vcl.ComCtrls,
  Vcl.StdCtrls,
  Vcl.ExtCtrls;

type
  TfrmMain = class(TForm)
    sbMain: TStatusBar;
    edRead_01: TEdit;
    edRead_02: TEdit;
    edRead_03: TEdit;
    edRead_04: TEdit;
    edRead_05: TEdit;
    edWrite_01: TEdit;
    edWrite_02: TEdit;
    edSubscription_01: TEdit;
    edSubscription_02: TEdit;
    edSubscription_03: TEdit;
    edSubscription_04: TEdit;
    edSubscription_05: TEdit;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    memoLog: TMemo;
    Label4: TLabel;
    tmrPoll: TTimer;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure tmrPollTimer(Sender: TObject);
    procedure edWrite_01Change(Sender: TObject);
    procedure edWrite_02Change(Sender: TObject);
  private
    FLoaded: Boolean;
    FClient: PUA_Client;
    FNode_DynDouble: UA_NodeId;
    FNode_DynString: UA_NodeId;
    FNode_DynInt32: UA_NodeId;
    FNode_StaticString: UA_NodeId;
    FNode_StaticDouble: UA_NodeId;

    procedure AddLog(Text: string);
    procedure AddMonitoredItem(subId: UA_UInt32; const NodeId: PUA_NodeId; TargetEdit: TEdit; SamplingInterval: Double);
    procedure SetupSubscriptionAndItems;

  public
  end;

var
  frmMain: TfrmMain;

implementation

{$R *.dfm}

procedure SubscriptionCallback(
  client: PUA_Client;
  subId: UA_UInt32;
  subContext: Pointer;
  monId: UA_UInt32;
  monContext: Pointer;
  value: PUA_DataValue
  ); cdecl;
var
  edit: TEdit;
  v: PUA_Variant;
  d: Double;
  i: UA_Int32;
  s: string;
begin
  edit := TEdit(monContext);
  v := @value^.value;

  if UA_Variant_hasScalarType(v, @UA_TYPES[UA_TYPES_DOUBLE]) then begin
    d := PDouble(v^.data)^;
    TThread.Synchronize(nil,
      procedure begin
        edit.Text := FloatToStr(d);
      end);
    Exit;
  end;

  if UA_Variant_hasScalarType(v, @UA_TYPES[UA_TYPES_INT32]) then begin
    i := PUA_Int32(v^.data)^;
    TThread.Synchronize(nil,
      procedure begin
        edit.Text := IntToStr(i);
      end);
    Exit;
  end;

  if UA_Variant_hasScalarType(v, @UA_TYPES[UA_TYPES_STRING]) then begin
    s := UA_StringFromVariant(v^);
    TThread.Synchronize(nil,
      procedure begin
        edit.Text := s;
      end);
    Exit;
  end;
end;

procedure TfrmMain.FormCreate(Sender: TObject);
begin
  memoLog.Lines.Clear;
  Caption := Caption + strPlattform;

  tmrPoll.Interval := 100;

  FLoaded := LoadOpen62541;
  if FLoaded then begin
    sbMain.Panels[2].Text := Format('Open62541 %s', [UA_OPEN62541_VERSION]);
    AddLog('DLL geladen');
  end else begin
    sbMain.Panels[2].Text := strDllNotLoaded;
    AddLog('DLL nicht geladen');
  end;

  AddLog('');
end;

procedure TfrmMain.FormShow(Sender: TObject);
var
  StatusCode: UA_StatusCode;
begin
  if not FLoaded then Exit;

  FClient := UA_Client_new;
  StatusCode := UA_ClientConfig_setDefault(UA_Client_getConfig(FClient));

  AddLog('ClientConfig_setDefault Result = ' + string(UA_StatusCode_name(StatusCode)));

  if StatusCode <> UA_STATUSCODE_GOOD then begin
    AddLog('Clientconfig konnte nicht geladen werden  Result = ' + string(UA_StatusCode_name(StatusCode)));
    Exit;
  end;

  AddLog('Default Clientconfig geladen');
  AddLog('');

  StatusCode := OpcUaClientConnect(FClient, PAnsiChar(AnsiString(strOpcServerIP)));

  if StatusCode <> UA_STATUSCODE_GOOD then begin
    AddLog('Connect Result = ' + string(UA_StatusCode_name(StatusCode)));
    AddLog('Client nicht verbunden');
    Exit;
  end;
  AddLog('OpcUaConnect  Result = ' + string(UA_StatusCode_name(StatusCode)));
  AddLog('Client verbunden');
  AddLog('');
  sbMain.Panels[1].Text := Format('Verbunden mit %s', [strOpcServerIP]);

  FNode_DynDouble := UA_NODEID_STRING_ALLOC(3, 'Demo.Dynamic.Scalar.Double');
  FNode_DynString := UA_NODEID_STRING_ALLOC(3, 'Demo.Dynamic.Scalar.String');
  FNode_DynInt32 := UA_NODEID_STRING_ALLOC(3, 'Demo.Dynamic.Scalar.Int32');
  FNode_StaticString := UA_NODEID_STRING_ALLOC(3, 'Demo.Static.Scalar.String');
  FNode_StaticDouble := UA_NODEID_STRING_ALLOC(3, 'Demo.Static.Scalar.Double');

  SetupSubscriptionAndItems;

  tmrPoll.Enabled := True;
end;

procedure TfrmMain.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  tmrPoll.Enabled := False;
  CanClose := True;
end;

procedure TfrmMain.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  UA_NodeId_clear(FNode_DynDouble);
  UA_NodeId_clear(FNode_DynString);
  UA_NodeId_clear(FNode_DynInt32);
  UA_NodeId_clear(FNode_StaticString);
  UA_NodeId_clear(FNode_StaticDouble);

  UnloadOpen62541;
end;

procedure TfrmMain.AddLog(Text: string);
begin
  memoLog.Lines.Add(Text);
end;

procedure TfrmMain.edWrite_01Change(Sender: TObject);
var
  nidWrite1: UA_NodeId;
begin
  if not FLoaded then
    Exit;

  nidWrite1 := UA_NODEID_STRING_ALLOC(3, 'Demo.Static.Scalar.String');
  UA_Client_writeValueAttribute(FClient, nidWrite1, AnsiString(edWrite_01.Text));
end;

procedure TfrmMain.edWrite_02Change(Sender: TObject);
var
  nidWrite2: UA_NodeId;
  d: UA_Double;
begin
  if not FLoaded then
    Exit;

  nidWrite2 := UA_NODEID_STRING_ALLOC(3, 'Demo.Static.Scalar.Double');
  if TryStrToFloat(edWrite_02.Text, d, FormatSettings) then begin
    UA_Client_writeValueAttribute(FClient, nidWrite2, d);
  end;
end;

procedure TfrmMain.tmrPollTimer(Sender: TObject);
var
  v: UA_Variant;
  d: UA_Double;
  i: UA_Int32;
  Res: UA_StatusCode;
  StatusCode: UA_StatusCode;
begin
  tmrPoll.Enabled := False;
  try
    if not FLoaded then Exit;

    StatusCOde := UA_Client_run_iterate(FClient, 0);
    if StatusCode <> UA_STATUSCODE_GOOD then begin
      AddLog('');
      AddLog('Verbindung verloren -> Reconnect wird ausgefhrt');
      SetupSubscriptionAndItems;
      Exit;
    end;

    UA_Variant_init(v);
    Res := UA_Client_readValueAttribute(FClient, FNode_DynDouble, v);
    if (Res = UA_STATUSCODE_GOOD) and UA_Variant_hasScalarType(@v, @UA_TYPES[UA_TYPES_DOUBLE]) then begin
      d := PDouble(v.data)^;
      edRead_01.Text := FloatToStr(d);
    end;
    UA_Variant_clear(v);

    UA_Variant_init(v);
    Res := UA_Client_readValueAttribute(FClient, FNode_DynString, v);
    if (Res = UA_STATUSCODE_GOOD) and UA_Variant_hasScalarType(@v, @UA_TYPES[UA_TYPES_STRING]) then
      edRead_02.Text := UA_StringFromVariant(v);
    UA_Variant_clear(v);

    UA_Variant_init(v);
    Res := UA_Client_readValueAttribute(FClient, FNode_DynInt32, v);
    if (Res = UA_STATUSCODE_GOOD) and UA_Variant_hasScalarType(@v, @UA_TYPES[UA_TYPES_INT32]) then begin
      i := PInteger(v.data)^;
      edRead_03.Text := IntToStr(i);
    end;
    UA_Variant_clear(v);

    UA_Variant_init(v);
    Res := UA_Client_readValueAttribute(FClient, FNode_StaticString, v);
    if (Res = UA_STATUSCODE_GOOD) and UA_Variant_hasScalarType(@v, @UA_TYPES[UA_TYPES_STRING]) then
      edRead_04.Text := UA_StringFromVariant(v);
    UA_Variant_clear(v);

    UA_Variant_init(v);
    Res := UA_Client_readValueAttribute(FClient, FNode_StaticDouble, v);
    if (Res = UA_STATUSCODE_GOOD) and UA_Variant_hasScalarType(@v, @UA_TYPES[UA_TYPES_DOUBLE]) then begin
      d := PDouble(v.data)^;
      edRead_05.Text := FloatToStr(d);
    end;
    UA_Variant_clear(v);

  finally
    tmrPoll.Enabled := True;
  end;
end;

procedure TfrmMain.AddMonitoredItem(subId: UA_UInt32; const NodeId: PUA_NodeId; TargetEdit: TEdit; SamplingInterval: Double);
var
  req: UA_MonitoredItemCreateRequest;
  result: UA_MonitoredItemCreateResult;
begin
  UA_MonitoredItemCreateRequest_init(req);
  req.itemToMonitor.nodeId := NodeId^;
  req.itemToMonitor.attributeId := UA_UInt32(UA_ATTRIBUTEID_VALUE);
  req.monitoringMode := UA_MONITORINGMODE_REPORTING;

  req.requestedParameters.samplingInterval := SamplingInterval;
  req.requestedParameters.queueSize := 1;
  req.requestedParameters.discardOldest := True;

  result := UA_Client_MonitoredItems_createDataChange(
    FClient,
    subId,
    UA_TIMESTAMPSTORETURN_BOTH,
    req,
    TargetEdit,
    @SubscriptionCallback,
    nil
    );

  if result.statusCode = UA_STATUSCODE_GOOD then
    AddLog('MonitoredItem erstellt: ID = ' + result.monitoredItemId.ToString)
  else
    AddLog('Fehler beim Erstellen eines MonitoredItems');
end;

procedure TfrmMain.SetupSubscriptionAndItems;
var
  subReq: UA_CreateSubscriptionRequest;
  subResp: UA_CreateSubscriptionResponse;
begin
  AddLog('');
  AddLog('Erstelle Subscription...');

  UA_CreateSubscriptionRequest_init(subReq);

  subReq.requestedPublishingInterval := 100.0;
  subReq.requestedLifetimeCount := 1000;
  subReq.requestedMaxKeepAliveCount := 10;
  subReq.maxNotificationsPerPublish := 0;
  subReq.publishingEnabled := True;
  subReq.priority := 0;

  subResp := UA_Client_Subscriptions_create(
    FClient,
    subReq,
    Self,
    nil,
    nil
    );

  if subResp.responseHeader.serviceResult <> UA_STATUSCODE_GOOD then begin
    AddLog('Subscription konnte nicht erstellt werden');
    Exit;
  end else begin
    AddLog('Subscription erstellt mit ID = ' + string(UA_StatusCode_name(subResp.responseHeader.serviceResult)));
    AddLog('');
  end;

  AddMonitoredItem(subResp.subscriptionId, @FNode_DynDouble, edSubscription_01, 100.0);
  AddMonitoredItem(subResp.subscriptionId, @FNode_DynString, edSubscription_02, 250.0);
  AddMonitoredItem(subResp.subscriptionId, @FNode_DynInt32, edSubscription_03, 500.0);
  AddMonitoredItem(subResp.subscriptionId, @FNode_StaticString, edSubscription_04, 100.0);
  AddMonitoredItem(subResp.subscriptionId, @FNode_StaticDouble, edSubscription_05, 200.0);

end;

end.

